//
//  MCRestRequest.h
//  MCFoundation
//
//  Created by Thomas Bartelmess on 10-02-03.
//  Copyright 2010 Marketcircle. All rights reserved.
//

#import <Foundation/Foundation.h>
/*!
 @enum		MCHTTPRequestType
 @abstract   Represents the HTTP request method
 @discussion
 @constant   MCRequestTypePOST Represents the POST HTTP Request method
 @constant   MCRequestTypeGET Represents the GET HTTP Request method
 @constant   MCRequestTypePUT Represents the PUT HTTP Request method
 @constant   MCRequestTypeDELETE Represents the DELETE HTTP Request method
 @constant   MCRequestTypeTRACE Represents the TRACE HTTP Request method
 @constant   MCRequestTypeOPTIONS Represents the OPTIONS HTTP Request method
 @constant   MCRequestTypeCONNECT Represents the CONNECT HTTP Request method
 */

typedef enum {
	MCHTTPRequestTypePOST = 0,
	MCHTTPRequestTypeGET = 1,
	MCHTTPRequestTypePUT = 2,
	MCHTTPRequestTypeDELETE = 3,
	MCHTTPRequestTypeTRACE = 4,
	MCHTTPRequestTypeOPTIONS = 5,
	MCHTTPRequestTypeCONNECT = 6
} MCHTTPRequestType;

/*!
 @enum		MCHTTPContentType
 @abstract   Represents the Content-Type header field in the HTTP Request
 @discussion
 @constant   MCHTTPContentTypeJSON represents the Content-Type application/JSON
 @constant   MCHTTPContentTypeXML represents the Content-Type application/XML
 @constant   MCHTTPContentTypePlainText represents the Content-Type text/plain
 */

typedef enum {
	MCHTTPContentTypeJSON = 0,
	MCHTTPContentTypeXML = 1,
	MCHTTPContentTypePlainText = 3,
	MCHTTPContentTypeHTML = 2,
	MCHTTPContentTypePlainXML = 4,
} MCHTTPContentType;

/*!
 @enum MCHTTPResult
 @abstract   Represents the result code of a HTTP Request
 @discussion The assignd number should represent the HTTP Result
 */


typedef enum {
	MCHTTPResultOK = 200,
	MCHTTPResultCreated = 201,
	MCHTTPResultDeleted = 204,
	MCHTTPResultBadRequest = 400,
	MCHTTPResultForbidden = 401,
	MCHTTPResultConflictOrDuplicate = 409,
	MCHTTPResultGone = 410,
	MCHTTPResultNotImplemented = 501,
	MCHTTPResultThrottled = 503
} MCHTTPResult;

/*!
 @class		MCRestRequest
 @abstract   Class to perform a RESTful HTTP request on a remote server
 @discussion MCRestRequest uses a NSURLConnection to perform the request. However all delegate functions of NSURLConnection are handled by MCRestRequest
 
 */

@class MCRestRequest;
@protocol MCRestRequestDelegate<NSObject>

@optional
- (void)restRequest:(MCRestRequest *)request didFinishSuccessfulWithResult:(NSDictionary *)info;
- (void)restRequest:(MCRestRequest *)request failedWithError:(NSError *)error;

@end


@interface MCRestRequest : NSObject {
	
	SEL successSelector;
	SEL failSelector;
	SEL responseSelector;
	id delegate;
	id _parameterDatasource;
	MCHTTPRequestType requestType;
	MCHTTPContentType contentType;
	
	NSURL *url;
	NSString *httpBody;
	
	NSMutableDictionary *responseHandlingInformations;
	
	NSString * requestUUID;
	
@private
	NSURLConnection * urlConnection;
	NSMutableURLRequest *request;
	NSMutableData *_data;
	NSURLResponse *_response;
	NSMutableDictionary * _parameters;
	NSString * _urlParameter;
}

@property (nonatomic, assign) id delegate;
@property (nonatomic, assign) id parameterDatasource;
@property (nonatomic, readwrite) SEL successSelector;
@property (nonatomic, readwrite) SEL failSelector;
@property (nonatomic, readwrite) SEL responseSelector;

@property (nonatomic, readwrite) MCHTTPRequestType requestType;
@property (nonatomic, readwrite) MCHTTPContentType contentType;
@property (nonatomic, retain) NSURL *url;
@property (nonatomic, retain) NSString *requestBody;
@property (nonatomic, retain) NSMutableDictionary *responseHandlingInformations;

@property (nonatomic, retain) NSMutableURLRequest *request;
@property (nonatomic, retain) NSURLConnection * urlConnection;
@property (nonatomic, retain) NSString * urlParameter;

@property (nonatomic, retain) NSString * requestUUID;

/*!
 @function	initWithConfigurationFile:
 @abstract   Returns an MCRestReqest object initialized with the configuration passed a property list file
 @param      path path to a property list file with a configuration for the MCRestRequest
 @param      baseURL base URL to which URL in the configuration is relative (do not provide a baseURL if the configuration's URL is absolute)
 @result     An configured MCRestReqest object
 */
- (id)initWithConfigurationFile:(NSString *)path baseURL:(NSURL *)baseURL;
- (id)initWithConfigurationFile:(NSString *)path;

/*!
 @function	initWithConfigurationFile:parameterDataSource:
 @abstract   Returns an MCRestReqest object initialized with the configuration passed a property list file and a parameter datasource
 @param      path path to a property list file with a configuration for the MCRestRequest
 @param      baseURL base URL to which URL in the configuration is relative (do not provide a baseURL if the configuration's URL is absolute)
 @param      parameterDataSource a object that can provide the required parameters
 @result     An configured MCRestReqest object
 */
- (id)initWithConfigurationFile:(NSString *)path baseURL:(NSURL *)baseURL parameterDataSource:(id)parameterDatasouce;
- (id)initWithConfigurationFile:(NSString *)path parameterDataSource:(id)parameterDatasouce;

/*!
 @function	initWithConfigurationFile:parameterDict:
 @abstract   Returns an initalized MCRestReqest object
 @param      path path to a property list file with a configuration for the MCRestRequest
 @param      baseURL base URL to which URL in the configuration is relative (do not provide a baseURL if the configuration's URL is absolute)
 @param      parameters Dictionary with parameters for the request.
 @result     An configured MCRestReqest object
 */
- (id)initWithConfigurationFile:(NSString *)path baseURL:(NSURL *)baseURL parameters:(NSDictionary *)parameters;
- (id)initWithConfigurationFile:(NSString *)path parameters:(NSDictionary *)parameters;

/*!
 @function	initWithConfigurationFile:urlParameter:
 @abstract   Returns an initalized MCRestReqest object
 @param      path path to a property list file with a configuration for the MCRestRequest
 @param      baseURL base URL to which URL in the configuration is relative (do not provide a baseURL if the configuration's URL is absolute)
 @param      urlParameter string that is appened to the url to pass additional parameters
 @result     An configured MCRestReqest object
 */
- (id)initWithConfigurationFile:(NSString *)path baseURL:(NSURL *)baseURL urlParameter:(NSString *)aUrlParameter;
- (id)initWithConfigurationFile:(NSString *)path urlParameter:(NSString *)aUrlParameter;

/*!
 @function	initWithConfigurationFile:parameterDataSource:parameters
 @abstract   Returns an initalized MCRestReqest object
 @param      path path to a property list file with a configuration for the MCRestRequest
 @param      baseURL base URL to which URL in the configuration is relative (do not provide a baseURL if the configuration's URL is absolute)
 @param      parameters Dictionary with parameters for the request.
 
 @result     An configured MCRestReqest object
 */
- (id)initWithConfigurationFile:(NSString *)path
						baseURL:(NSURL *)baseURL
			parameterDataSource:(id)parameterDatasouce
					 parameters:(NSDictionary *)parameters;
- (id)initWithConfigurationFile:(NSString *)path
			parameterDataSource:(id)parameterDatasouce
					 parameters:(NSDictionary *)parameters;

/*!
 @function	initWithConfigurationFile:parameterDict:
 @abstract   Returns an initalized MCRestReqest object
 @param      path path to a property list file with a configuration for the MCRestRequest
 @param      baseURL base URL to which URL in the configuration is relative (do not provide a baseURL if the configuration's URL is absolute)
 @param      parameters Dictionary with parameters for the request.
 @param      urlParameter string that is appened to the url to pass additional parameters
 
 @result     An configured MCRestReqest object
 */
- (id)initWithConfigurationFile:(NSString *)path
						baseURL:(NSURL *)baseURL
					 parameters:(NSDictionary *)parameters
				   urlParameter:(NSString *)aUrlParameter;
- (id)initWithConfigurationFile:(NSString *)path
					 parameters:(NSDictionary *)parameters
				   urlParameter:(NSString *)aUrlParameter;

/*!
 @function	initWithConfigurationFile:parameterDict:
 @abstract   Returns an initalized MCRestReqest object
 @param      path path to a property list file with a configuration for the MCRestRequest
 @param      baseURL base URL to which URL in the configuration is relative (do not provide a baseURL if the configuration's URL is absolute)
 @param      parameterDataSource a object that can provide the required parameters
 @param      parameters Dictionary with parameters for the request.
 @param      urlParameter string that is appened to the url to pass additional parameters
 
 @result     An configured MCRestReqest object
 */
- (id)initWithConfigurationFile:(NSString *)path
						baseURL:(NSURL *)baseURL
			parameterDataSource:(id)parameterDatasouce
					 parameters:(NSDictionary *)parameters
				   urlParameter:(NSString *)aUrlParameter;

- (id)initWithConfigurationFile:(NSString *)path
			parameterDataSource:(id)parameterDatasouce
					 parameters:(NSDictionary *)parameters
				   urlParameter:(NSString *)aUrlParameter;


/*!
 @function	initWithConfiguration:
 @abstract   Returns an initalized MCRestReqest object
 @param		configuration dictionary with config settings for MCRestRequest
 @result     A initalized and configured MCRestReqest object
 */
- (id)initWithConfiguration:(NSDictionary *)configuration;

/*!
 @function	initWithConfiguration:
 @abstract   Returns an initalized MCRestReqest object
 @param		configuration dictionary with config settings for MCRestRequest
 
 @result     A initalized and configured MCRestReqest object
 */
- (id)initWithConfiguration:(NSDictionary *)configuration
		parameterDataSource:(id)parameterDatasouce;

/*!
 @function	initWithConfiguration:parameters
 @abstract   Returns an initalized MCRestReqest object
 @param		configuration dictionary with config settings for MCRestRequest
 @param      parameterDataSource a object that can provide the required parameters
 @param      parameters Dictionary with parameters for the request.
 
 @result     A initalized and configured MCRestReqest object
 */
- (id)initWithConfiguration:(NSDictionary *)configuration
                 parameters:(NSDictionary *)parameters;

/*!
 @function	initWithConfiguration:urlParameter
 @abstract   Returns an initalized MCRestReqest object
 @param		configuration dictionary with config settings for MCRestRequest
 @param      urlParameter string that is appened to the url to pass additional parameters
 
 @result     A initalized and configured MCRestReqest object
 */
- (id)initWithConfiguration:(NSDictionary *)configuration
			   urlParameter:(NSString *)aUrlParameter;

/*!
 @function	initWithConfiguration:parameterDataSource:parameters:
 @abstract   Returns an initalized MCRestReqest object
 @param		configuration dictionary with config settings for MCRestRequest
 @param      parameterDataSource a object that can provide the required parameters
 
 @result     A initalized and configured MCRestReqest object
 */
- (id)initWithConfiguration:(NSDictionary *)configuration
		parameterDataSource:(id)parameterDatasouce
                 parameters:(NSDictionary *)parameters;

/*!
 @function	initWithConfiguration:parameterDataSource:parameters:
 @abstract   Returns an initalized MCRestReqest object
 @param		configuration dictionary with config settings for MCRestRequest
 @param      parameterDataSource a object that can provide the required parameters
 @param      parameters Dictionary with parameters for the request.
 
 @result     A initalized and configured MCRestReqest object
 */
- (id)initWithConfiguration:(NSDictionary *)configuration
		parameterDataSource:(id)parameterDatasouce
				 parameters:(NSDictionary *)parameters;

/*!
 @function	initWithConfiguration:parameters:urlParameter:
 @abstract   Returns an initalized MCRestReqest object
 @param		configuration dictionary with config settings for MCRestRequest
 @param      parameters Dictionary with parameters for the request.
 @param      urlParameter string that is appened to the url to pass additional parameters
 
 @result     A initalized and configured MCRestReqest object
 */
- (id)initWithConfiguration:(NSDictionary *)configuration
				 parameters:(NSDictionary *)parameters
			   urlParameter:(NSString *)aUrlParameter;

/*!
 @function	initWithConfiguration:parameterDataSource:parameters:urlParameter:
 @abstract   Returns an initalized MCRestReqest object
 @param		configuration dictionary with config settings for MCRestRequest
 @param      parameterDataSource a object that can provide the required parameters
 @param      parameters Dictionary with parameters for the request.
 @param      urlParameter string that is appened to the url to pass additional parameters
 
 @result     A initalized and configured MCRestReqest object
 */

- (id)initWithConfiguration:(NSDictionary *)configuration
		parameterDataSource:(id)parameterDatasouce
				 parameters:(NSDictionary *)parameters
			   urlParameter:(NSString *)aUrlParameter;


/*!
 @method     start
 @abstract   Start performing the request.
 */
- (void)start;

- (void)parseDataSourceParametersWithConfigurations:(NSArray *)parameterConfigurations;
@end

@interface NSObject (MCRestRequestParameterDatasource)

- (NSString *)parameterForBaseURLString:(NSString *)anURL;

@end

